Вот минималистичный, практичный план под твои условия — без усложнённой безопасности, в формате `.md`.
Ты используешь MCP-сервер для личного применения, с ручной настройкой логина/пароля, а защита ограничена только таймаутом, лимитом строк и размера текста.

---

# Oracle DB MCP Tool — Development Plan

## Overview

**Goal:**
Создать простой MCP-tool для VS Code, который через 32-битный Python (`D:\Dropbox\dev\python\python.exe`) позволяет выполнять SQL и PL/SQL в Oracle Database, просматривать метаданные, создавать и тестировать процедуры.

**Scope:**

* Без RAG, кэширования и хранения знаний.
* Без сложных мер безопасности.
* Только: `timeout`, `max_rows`, `max_text_size`.
* Подключение к Oracle — через логин/пароль, прописанные в окружении.

---

## 1. Architecture

**VS Code (MCP client)**
↓
**MCP server (Python)**
↓
**Oracle DB via cx_Oracle (32-bit)**

### Core Features

* **Метаданные:**

  * `list_schemas`
  * `list_objects`
  * `describe_object`
  * `get_ddl`
  * `explain_plan`
* **Выполнение:**

  * `query` — безопасный SELECT
  * `exec_plsql` — выполнение PL/SQL блока
  * `create_or_replace_proc` — создание процедуры
  * `call_proc` — вызов процедуры
  * `run_test` — простая проверка выполнения
* **Служебное:**

  * `ping`, `session_info`

---

## 2. Environment Setup

### Python

* Используется **Python 32-bit**:

  ```
  D:\Dropbox\dev\python\python.exe
  ```
* Установить пакеты:

  ```
  pip install mcp cx_Oracle==8.3.0 attrs
  ```

### Oracle Instant Client

* Скачать **32-битный Oracle Instant Client**.
* Добавить в `PATH`.

### Environment Variables

```bash
ORACLE_USER=v8live
ORACLE_PASS=live
ORACLE_DSN=10.0.0.18:1521/v8
MCP_MAX_ROWS=500
MCP_MAX_TEXT_SIZE=5000
MCP_TIMEOUT_S=30
```

---

## 3. Repository Structure

```

├─ README.md
├─ requirements.txt
├─ server/
│  ├─ main.py                 # entrypoint for MCP
│  ├─ mcp_server.py           # registers tools
│  ├─ oracle/
│  │  ├─ pool.py              # connection pool
│  │  ├─ exec.py              # query/exec/explain
│  │  ├─ meta.py              # metadata tools
│  │  └─ tests.py             # unit test helpers
│  └─ util/
│     ├─ json.py              # serialization of DATE, TIMESTAMP
│     └─ log.py
└─ .vscode/
   └─ mcp.json
```

---

## 4. MCP Tools Definition

### `db.query`

Executes SQL (mostly SELECT).
Parameters:

```json
{ "sql": "select * from dual", "binds": {}, "max_rows": 500, "timeout_s": 30 }
```

Returns JSON with:

```json
{ "columns": ["DUMMY"], "rows": [["X"]], "truncated": false }
```

### `db.exec_plsql`

Executes PL/SQL block (autocommit by default).

### `db.get_ddl`

Returns DDL for object:

```sql
SELECT DBMS_METADATA.GET_DDL(object_type, object_name, owner) FROM DUAL
```

### `db.explain`

Runs `EXPLAIN PLAN FOR` + `DBMS_XPLAN.DISPLAY()`.

### `db.create_or_replace_proc`

Replaces a stored procedure from text source.

### `db.call_proc`

Calls procedure with parameters.

### `db.list_objects`

Lists objects from `ALL_OBJECTS`.

### `db.describe_object`

Shows columns (for tables) or arguments (for procedures).

---

## 5. Implementation Details

### Timeout

* Applied via `conn.callTimeout = timeout_s * 1000`.

### Max Rows

* Limit result fetching:

  ```python
  cur.arraysize = min(max_rows, 1000)
  for i, row in enumerate(cur):
      if i >= max_rows: break
  ```

### Max Text Size

* Truncate long strings before serialization.

### JSON Serialization

* Replace `datetime`/`decimal` with strings.

### Errors

* Return `{"error": "message"}` in JSON.

---

## 6. VS Code Integration

`.vscode/mcp.json`:

```json
{
  "servers": {
    "oracle-db": {
      "command": "D:\\Dropbox\\dev\\python\\python.exe",
      "args": ["-u", "server\\main.py"],
      "env": {
        "ORACLE_DSN": "10.0.0.18:1521/v8",
        "ORACLE_USER": "v8live",
        "ORACLE_PASS": "live"
      },
      "enabled": true
    }
  }
}
```

---

## 7. Development Stages

### Stage 1 — Setup and Connectivity

* [ ] Создать venv и активировать его.
* [ ] Установить cx_Oracle и проверить подключение.
* [ ] Реализовать `OraclePool` с `SessionPool`.
* [ ] Добавить `ping()` и `session_info()`.

### Stage 2 — Query and Metadata

* [ ] Реализовать `db.query`.
* [ ] Добавить `list_schemas`, `list_objects`.
* [ ] Добавить `describe_object`.
* [ ] Добавить `get_ddl`, `explain`.

### Stage 3 — PL/SQL Operations

* [ ] `exec_plsql`, `create_or_replace_proc`, `call_proc`.
* [ ] Простая функция `run_test()` для проверки процедуры.

### Stage 4 — MCP Integration

* [ ] Зарегистрировать все инструменты в `mcp_server.py`.
* [ ] Проверить, что VS Code видит MCP-сервер.
* [ ] Проверить выполнение SQL и возврат результата.

### Stage 5 — UX Improvements

* [ ] Форматировать вывод (align columns, truncate text).
* [ ] Добавить опциональный `timing_ms` в ответах.
* [ ] Добавить минимальные логи в `util/log.py`.

---

## 8. Example Snippets

**Query Example**

```python
def query(sql, binds=None, max_rows=500, timeout_s=30):
    with pool.acquire() as conn:
        conn.callTimeout = timeout_s * 1000
        cur = conn.cursor()
        cur.arraysize = min(max_rows, 1000)
        cur.execute(sql, binds or {})
        cols = [d[0] for d in cur.description]
        rows = []
        for i, r in enumerate(cur):
            if i >= max_rows: break
            rows.append([str(x)[:5000] if isinstance(x, str) else x for x in r])
        return {"columns": cols, "rows": rows}
```

**Explain Example**

```python
def explain(sql, binds=None):
    with pool.acquire() as conn:
        cur = conn.cursor()
        cur.execute("EXPLAIN PLAN FOR " + sql, binds or {})
        cur.execute("SELECT * FROM TABLE(DBMS_XPLAN.DISPLAY())")
        plan = "\n".join(r[0] for r in cur)
        return {"plan": plan}
```

---

## 9. Deliverables

* ✅ **Functional MCP server** callable from VS Code.
* ✅ **Query, metadata, DDL, and procedure tools** implemented.
* ✅ **Timeout, max_rows, and text trimming** enforced.
* ✅ **README.md** with setup instructions.

---

## 10. Future Extensions (Optional)

* [ ] Автоматические тесты (`pytest`) для MCP-tools.
* [ ] Генерация DDL-скриптов по схемам.
* [ ] Кэширование метаданных.
* [ ] Интеграция с твоим RAG-сервисом позже.

---

Хочешь — я сразу сгенерирую шаблон `server/main.py` и `mcp_server.py`, чтобы ты мог запустить первую версию в VS Code.
